// Game_Music_Emu 0.4.0. http://www.slack.net/~ant/

#include "Music_Emu.h"

#include <string.h>

/* Copyright (C) 2003-2006 Shay Green. This module is free software; you
can redistribute it and/or modify it under the terms of the GNU Lesser
General Public License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version. This
module is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for
more details. You should have received a copy of the GNU Lesser General
Public License along with this module; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA */

#include "blargg_source.h"

int const stereo = 2; // number of channels for stereo
int const silence_max = 6; // seconds
int const silence_threshold = 0x10;
long const fade_block_size = 512;
int const fade_shift = 8; // fade ends with gain at 1.0 / (1 << fade_shift)

Music_Emu::equalizer_t const Music_Emu::tv_eq = { -8.0, 180 };

void Music_Emu::set_track_vars( int track )
{
	current_track_   = track;
	error_count_     = 0;
	out_time         = 0;
	emu_time         = 0;
	emu_track_ended_ = false;
	track_ended_     = false;
	fade_start       = LONG_MAX / 2 + 1;
	fade_step        = 1;
	silence_time     = 0;
	silence_count    = 0;
	buf_remain       = 0;
}

void Music_Emu::unload()
{
	track_count_ = 0;
	voice_count_ = 0;
	set_track_vars( -1 );
}

Music_Emu::Music_Emu()
{
	sample_rate_ = 0;
	
	// defaults
	silence_detection = true;
	equalizer_.treble = -1.0;
	equalizer_.bass   = 60;
	mute_mask_        = 0;
	
	unload();
}

Music_Emu::~Music_Emu()
{
}

blargg_err_t Music_Emu::set_sample_rate( long rate )
{
	assert( !sample_rate_ ); // sample rate can't be changed once set
	RETURN_ERR( set_sample_rate_( rate ) );
	RETURN_ERR( buf.resize( buf_size ) );
	sample_rate_ = rate;
	return 0;
}

blargg_err_t Music_Emu::load_file( const char* path )
{
	Vfs_File_Reader in;
	RETURN_ERR( in.open( path ) );
	return load( in );
}

const char** Music_Emu::voice_names() const
{
	static const char* names [] = {
		"Voice 1", "Voice 2", "Voice 3", "Voice 4",
		"Voice 5", "Voice 6", "Voice 7", "Voice 8"
	};
	return names;
}

void Music_Emu::start_track( int track )
{
	assert( sample_rate() ); // set_sample_rate() must have been called first
	assert( (unsigned) track <= (unsigned) track_count() );
	
	set_track_vars( track );
	start_track_( track );
	
	if ( silence_detection )
	{
		// play until non-silence or end of track
		for ( long end = 40 * stereo * sample_rate(); emu_time < end; )
		{
			fill_buf();
			if ( buf_remain | emu_track_ended_ )
				break;
		}
		
		emu_time      = buf_remain;
		out_time      = 0;
		silence_time  = 0;
		silence_count = 0;
	}
}

long Music_Emu::msec_to_samples( long msec ) const
{
	long sec = msec / 1000;
	msec -= sec * 1000;
	return (sec * sample_rate() + msec * sample_rate() / 1000) * stereo;
}

// Tell/Seek

long Music_Emu::tell() const
{
	long rate = sample_rate() * stereo;
	long sec = out_time / rate;
	return sec * 1000 + (out_time - sec * rate) * 1000 / rate;
}

void Music_Emu::seek( long msec )
{
	long time = msec_to_samples( msec );
	if ( time < out_time )
		start_track( current_track_ );
	skip( time - out_time );
}

void Music_Emu::skip( long count )
{
	out_time += count;
	
	// remove from silence and buf first
	{
		long n = min( count, silence_count );
		silence_count -= n;
		count -= n;
		
		n = min( count, buf_remain );
		buf_remain -= n;
		count -= n;
	}
	
	emu_time += count;
	skip_( count );
}

void Music_Emu::skip_( long count )
{
	// for long skip, mute sound
	const long threshold = 30000;
	if ( count > threshold )
	{
		int saved_mute = mute_mask_;
		mute_voices( ~0 );
		
		while ( count > threshold / 2 )
		{
			play_( buf_size, buf.begin() );
			count -= buf_size;
		}
		
		mute_voices( saved_mute );
	}
	
	while ( count )
	{
		long n = buf_size;
		if ( n > count )
			n = count;
		count -= n;
		play_( n, buf.begin() );
	}
}

// Fading

void Music_Emu::set_fade( long start_msec, long length_msec )
{
	fade_step = sample_rate() * length_msec / (fade_block_size * fade_shift * 1000 / stereo);
	fade_start = msec_to_samples( start_msec );
}

// unit / pow( 2, (double) x / step )
static int int_log( long x, int step, int unit )
{
	int shift = x / step;
	int fraction = (x - shift * step) * unit / step;
	return ((unit - fraction) + (fraction >> 1)) >> shift;
}

void Music_Emu::handle_fade( long out_count, sample_t* out )
{
	for ( int i = 0; i < out_count; i += fade_block_size )
	{
		int const shift = 14;
		int const unit = 1 << shift;
		int gain = int_log( (out_time + i - fade_start) / fade_block_size,
				fade_step, unit );
		if ( gain < (unit >> fade_shift) )
			track_ended_ = true;
		
		sample_t* io = &out [i];
		for ( int count = min( fade_block_size, out_count - i ); count; --count )
		{
			*io = (*io * gain) >> shift;
			++io;
		}
	}
}

// Silence detection

void Music_Emu::emu_play( long count, sample_t* out )
{
	check( current_track_ >= 0 );
	emu_time += count;
	if ( current_track_ >= 0 && !track_ended_ )
		play_( count, out );
	else
		memset( out, 0, count * sizeof *out );
}

// Number of consecutive silent samples at end
static long count_silence( Music_Emu::sample_t* begin, long size )
{
	int first = *begin;
	*begin = silence_threshold; // sentinel
	Music_Emu::sample_t* p = begin + size;
	while ( (unsigned) (*--p + silence_threshold / 2) <= (unsigned) silence_threshold ) { }
	*begin = first;
	return size - (p - begin);
}

// Fill internal buffer and check it for silence
void Music_Emu::fill_buf()
{
	assert( !buf_remain );
	emu_play( buf_size, buf.begin() );
	long silence = count_silence( buf.begin(), buf_size );
	if ( silence < buf_size )
	{
		silence_time = emu_time - silence;
		buf_remain = buf_size;
	}
	else
	{
		silence_count += buf_size;
	}
}

void Music_Emu::play( long out_count, sample_t* out )
{
	assert( out_count % stereo == 0 );
	assert( emu_time >= out_time );
	
	// nifty graph of how far ahead we are when searching for silence
	//dprintf( "%*s \n", int ((emu_time - out_time) * 7 / sample_rate()), "*" );
	
	long pos = 0;
	if ( silence_count )
	{
		// during a run of silence, run emulator at 6x speed so it gets ahead
		long ahead_time = 6 * (out_time + out_count - silence_time) + silence_time;
		while ( emu_time < ahead_time && !(buf_remain | track_ended_ | emu_track_ended_) )
			fill_buf();
		
		// fill with silence
		pos = min( silence_count, out_count );
		memset( out, 0, pos * sizeof *out );
		silence_count -= pos;
		
		if ( emu_time - silence_time > silence_max * stereo * sample_rate() )
		{
			track_ended_ = true;
			silence_count = 0;
			buf_remain = 0;
		}
	}
	
	if ( buf_remain )
	{
		// empty silence buf
		long n = min( buf_remain, out_count - pos );
		memcpy( &out [pos], &buf [buf_size - buf_remain],
				n * sizeof *out );
		buf_remain -= n;
		pos += n;
	}
	
	// generate remaining samples normally
	long remain = out_count - pos;
	if ( remain )
	{
		emu_play( remain, out + pos );
		track_ended_ |= emu_track_ended_;
		
		if ( silence_detection || out_time > fade_start )
		{
			// check end for a new run of silence
			long silence = count_silence( out + pos, remain );
			if ( silence < remain )
				silence_time = emu_time - silence;
			
			if ( emu_time - silence_time >= buf_size )
				fill_buf(); // cause silence detection on next play()
		}
	}
	
	if ( out_time > fade_start )
		handle_fade( out_count, out );
	
	out_time += out_count;
}

// Track info

void track_info_t::init()
{
	track_count  = 1;
	
	length       = -1;
	loop_length  = -1;
	intro_length = -1;
	
	game [0]      = 0;
	song [0]      = 0;
	author [0]    = 0;
	copyright [0] = 0;
	comment [0]   = 0;
	dumper [0]    = 0;
	system [0]    = 0;
}

void track_info_t::copy_field( char* out, const char* in, int in_size )
{
	if ( !*out )
	{
		// remove spaces/junk from beginning
		while ( in_size && unsigned (*in - 1) <= ' ' - 1 )
		{
			in++;
			in_size--;
		}
		
		// truncate
		if ( in_size > max_field )
			in_size = max_field;
		
		// find terminator
		int len = 0;
		while ( len < in_size && in [len] )
			len++;
		
		// remove spaces/junk from end
		while ( len && unsigned (in [len - 1]) <= ' ' )
			len--;
		
		// copy
		out [len] = 0;
		memcpy( out, in, len );
		
		// strip out stupid fields that should have been left blank
		if ( !strcmp( out, "?" ) || !strcmp( out, "<?>" ) || !strcmp( out, "< ? >" ) )
			out [0] = 0;
	}
}

void track_info_t::copy_field( char* out, const char* in )
{
	copy_field( out, in, max_field );
}

blargg_err_t Music_Emu::track_info( track_info_t* out, int track ) const
{
	assert( (unsigned) track < track_count() );
	out->init();
	out->track_count = track_count();
	return out->error();
}

blargg_err_t Music_Emu::track_info( track_info_t* out ) const
{
	return track_info( out, current_track_ );
}

// C interface

const char* gme_read_track_info( gme_type_t type, track_info_t* out, int track,
		const char* data, long size )
{
	if ( !size )
	{
		Vfs_File_Reader in;
		RETURN_ERR( in.open( data ) );
		return type->read_track_info( in, out, track );
	}
	
	Mem_File_Reader in( data, size );
	return type->read_track_info( in, out, track );
}

Music_Emu* gme_new_emu( gme_type_t type, long rate )
{
	Music_Emu* me = type->new_emu();
	if ( me && !me->set_sample_rate( rate ) )
		return me;
	
	delete me;
	return 0;
}

const char* gme_load_file( Music_Emu* me, const char* data, long size )
{
	if ( !size )
		return me->load_file( data );
	
	Mem_File_Reader in( data, size );
	return me->load( in );
}

int gme_track_count( Music_Emu const* me ) { return me->track_count(); }

const char* gme_track_info( Music_Emu* me, track_info_t* out, int track )
{
	return me->track_info( out, track );
}

void gme_start_track( Music_Emu* me, int index ) { me->start_track( index ); }

void gme_set_fade( Music_Emu* me, long start_msec ) { me->set_fade( start_msec ); }

void gme_play( Music_Emu* me, long n, short* p ) { me->play( n, p ); }

long gme_tell( Music_Emu const* me ) { return me->tell(); }

void gme_seek( Music_Emu* me, long msec ) { me->seek( msec ); }

int gme_track_ended( Music_Emu const* me ) { return me->track_ended(); }

void gme_delete( Music_Emu* me ) { delete me; }
